<?php
/**
 * Plugin Name: Mackens Mini-Galleri
 * Plugin URI: https://example.com
 * Description: Kortkod som visar ett mini-galleri i filmremsestil. Bilder väljs per inlägg via en metabox. Placera galleriet där du vill med [macken_mini_gallery].
 * Version: 1.3.3
 * Author: Magasin Macken
 * Text Domain: mackens-mini-galleri
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'MMG_VERSION', '1.3.3' );
define( 'MMG_SLUG', 'mackens-mini-galleri' );
define( 'MMG_META_KEY', '_mmg_image_ids' );
define( 'MMG_URL', plugin_dir_url( __FILE__ ) );
define( 'MMG_PATH', plugin_dir_path( __FILE__ ) );

class Mackens_Mini_Galleri {

    public function __construct() {
        // Admin metabox
        add_action( 'add_meta_boxes', [ $this, 'add_metabox' ] );
        add_action( 'save_post', [ $this, 'save_metabox' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_assets' ] );

        // Shortcode
        add_shortcode( 'macken_mini_gallery', [ $this, 'shortcode_gallery' ] );
    }

    public function add_metabox() {
        add_meta_box(
            'mmg_metabox',
            __( 'Mini-galleri', 'mackens-mini-galleri' ),
            [ $this, 'render_metabox' ],
            'post',
            'normal',
            'default'
        );
    }

    public function render_metabox( $post ) {
        wp_nonce_field( 'mmg_save', 'mmg_nonce' );

        $ids = get_post_meta( $post->ID, MMG_META_KEY, true );
        $ids = is_string( $ids ) ? $ids : '';
        $ids_arr = array_filter( array_map( 'intval', explode( ',', $ids ) ) );

        // Preload thumbs
        $thumbs = [];
        foreach ( $ids_arr as $id ) {
            $src = wp_get_attachment_image_src( $id, 'thumbnail' );
            if ( $src ) {
                $alt = get_post_meta( $id, '_wp_attachment_image_alt', true );
                $thumbs[] = [
                    'id' => $id,
                    'src' => $src[0],
                    'alt' => $alt ? $alt : ''
                ];
            }
        }

        echo '<div class="mmg-wrap">';
        echo '<p>' . esc_html__( 'Välj eller ladda upp bilder till mini-galleriet. Dra för att ändra ordning.', 'mackens-mini-galleri' ) . '</p>';
        echo '<input type="hidden" id="mmg-image-ids" name="mmg_image_ids" value="' . esc_attr( implode( ',', $ids_arr ) ) . '">';
        echo '<div id="mmg-list" class="mmg-list">';

        foreach ( $thumbs as $t ) {
            echo '<div class="mmg-item" data-id="' . esc_attr( $t['id'] ) . '">';
            echo '<img src="' . esc_url( $t['src'] ) . '" alt="' . esc_attr( $t['alt'] ) . '"/>';
            echo '<button type="button" class="button-link mmg-remove" aria-label="' . esc_attr__( 'Ta bort', 'mackens-mini-galleri' ) . '">&times;</button>';
            echo '</div>';
        }

        echo '</div>'; // list
        echo '<p><button type="button" class="button button-primary" id="mmg-add">' . esc_html__( 'Lägg till bilder', 'mackens-mini-galleri' ) . '</button></p>';
        echo '</div>';
    }

    public function save_metabox( $post_id ) {
        if ( ! isset( $_POST['mmg_nonce'] ) || ! wp_verify_nonce( $_POST['mmg_nonce'], 'mmg_save' ) ) return;
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;

        $post_type = get_post_type( $post_id );
        if ( $post_type !== 'post' ) return;

        if ( ! current_user_can( 'edit_post', $post_id ) ) return;

        $ids = isset( $_POST['mmg_image_ids'] ) ? sanitize_text_field( wp_unslash( $_POST['mmg_image_ids'] ) ) : '';
        // normalize to comma-separated integers
        $ids_arr = array_filter( array_map( 'intval', array_map( 'trim', explode( ',', $ids ) ) ) );
        $value = implode( ',', $ids_arr );

        if ( $value ) {
            update_post_meta( $post_id, MMG_META_KEY, $value );
        } else {
            delete_post_meta( $post_id, MMG_META_KEY );
        }
    }

    public function admin_assets( $hook ) {
        $screen = get_current_screen();
        if ( $screen && $screen->base === 'post' ) {
            wp_enqueue_media();
            wp_enqueue_script( 'jquery-ui-sortable' );
            wp_enqueue_style( 'mmg-admin', MMG_URL . 'assets/css/admin.css', [], MMG_VERSION );
            wp_enqueue_script( 'mmg-admin', MMG_URL . 'assets/js/admin.js', [ 'jquery', 'jquery-ui-sortable' ], MMG_VERSION, true );
        }
    }

    public function shortcode_gallery( $atts, $content = null, $tag = '' ) {
        $atts = shortcode_atts(
            [
                'post_id' => 0,
            ],
            $atts,
            'macken_mini_gallery'
        );

        $post_id = intval( $atts['post_id'] );
        if ( ! $post_id ) {
            $post_id = get_the_ID();
        }
        if ( ! $post_id ) {
            $post_id = get_queried_object_id();
        }
        if ( ! $post_id ) {
            global $post;
            if ( $post instanceof WP_Post ) {
                $post_id = (int) $post->ID;
            }
        }

        if ( ! $post_id ) return '';

        $ids = get_post_meta( $post_id, MMG_META_KEY, true );
        if ( ! $ids ) return '';

        $ids_arr = array_filter( array_map( 'intval', explode( ',', $ids ) ) );
        if ( empty( $ids_arr ) ) return '';

        // Enqueue frontend styles once
        wp_enqueue_style( 'mmg-frontend', MMG_URL . 'assets/css/frontend.css', [], MMG_VERSION );
        wp_enqueue_script( 'mmg-frontend', MMG_URL . 'assets/js/frontend.js', [], MMG_VERSION, true );

        $out = '<div class="mmg-filmstrip" aria-label="' . esc_attr__( 'Fler bilder', 'mackens-mini-galleri' ) . '">'
            . '<button type="button" class="mmg-nav mmg-prev" aria-label="Föregående">‹</button>'
            . '<div class="mmg-track" role="list">';
        foreach ( $ids_arr as $id ) {
            $thumb = wp_get_attachment_image_src( $id, 'thumbnail' );
            if ( ! $thumb ) continue;

            $thumb_url = esc_url( $thumb[0] );
            $full_url_raw = wp_get_attachment_url( $id );
            if ( ! $full_url_raw ) {
                $full_url_raw = wp_get_attachment_image_url( $id, 'full' );
            }
            if ( ! $full_url_raw ) {
                $full_url_raw = $thumb[0];
            }
            $full_url  = esc_url( $full_url_raw );
            $alt       = get_post_meta( $id, '_wp_attachment_image_alt', true );
            $alt       = $alt ? $alt : '';

            $out .= '<a class="mmg-link" href="' . $full_url . '" data-full="' . $full_url . '">';
            $out .= '<img class="mmg-thumb" src="' . $thumb_url . '" alt="' . esc_attr( $alt ) . '" loading="lazy" />';
            $out .= '</a>';
        }
        $out .= '</div>' . '<button type="button" class="mmg-nav mmg-next" aria-label="Nästa">›</button>' . '</div>';

        return $out;
    }
}

new Mackens_Mini_Galleri();
