<?php
/**
 * Plugin Name: Macken WXR Export
 * Description: Exportera alla inlägg (oavsett status) i en vald kategori till en WXR/XML som kan importeras i WordPress. Finns under Verktyg → Exportera inlägg (WXR).
 * Version: 1.0.2
 * Author: Magasin Macken
 * License: GPL2+
 */

if (!defined('ABSPATH')) exit;

class Macken_WXR_Export {
    const CAP = 'manage_options';
    const NONCE_ACTION = 'macken_wxr_export_action';
    const SLUG = 'macken-wxr-export';

    public function __construct() {
        add_action('admin_menu', [$this, 'add_menu']);
        // Separate download endpoint to avoid mixing with admin page output
        add_action('admin_post_mwxr_export', [$this, 'handle_export']);
    }

    public function add_menu() {
        add_management_page(
            'Exportera inlägg (WXR)',
            'Exportera inlägg (WXR)',
            self::CAP,
            self::SLUG,
            [$this, 'render_page']
        );
    }

    public function render_page() {
        if (!current_user_can(self::CAP)) {
            wp_die(__('Otillräckliga rättigheter.', 'macken-wxr-export'));
        }

        // Build category dropdown (include empty categories)
        $cats = get_terms([
            'taxonomy'   => 'category',
            'hide_empty' => false,
            'orderby'    => 'name',
            'order'      => 'ASC',
        ]);

        $action_url = admin_url('admin-post.php');
        ?>
        <div class="wrap">
            <h1>Exportera inlägg (WXR)</h1>
            <p>Skapar en WXR/XML som laddas ned direkt och kan importeras via <em>Verktyg → Importera</em> i WordPress. Exporten innehåller alla inlägg i vald kategori, oavsett status. Som standard utesluts papperskorgen.</p>

            <form method="post" action="<?php echo esc_url($action_url); ?>">
                <input type="hidden" name="action" value="mwxr_export" />
                <?php wp_nonce_field(self::NONCE_ACTION); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="category_id">Kategori</label></th>
                        <td>
                            <select id="category_id" name="category_id" required>
                                <option value="">— Välj kategori —</option>
                                <?php
                                if (!is_wp_error($cats)) {
                                    foreach ($cats as $cat) {
                                        printf(
                                            '<option value="%d">%s (ID %d, slug %s)</option>',
                                            (int)$cat->term_id,
                                            esc_html($cat->name),
                                            (int)$cat->term_id,
                                            esc_html($cat->slug)
                                        );
                                    }
                                }
                                ?>
                            </select>
                            <p class="description">Listan visar alla kategorier, även sådana som bara innehåller utkast.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Inkludera papperskorg</th>
                        <td>
                            <label><input type="checkbox" name="include_trash" value="1"> Ja, ta med inlägg i papperskorgen</label>
                            <p class="description">Som standard ingår alla statusar utom papperskorg (trash).</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Exportera WXR'); ?>
            </form>
        </div>
        <?php
    }

    public function handle_export() {
        if (!current_user_can(self::CAP)) {
            wp_die(__('Otillräckliga rättigheter.', 'macken-wxr-export'));
        }
        if (!check_admin_referer(self::NONCE_ACTION)) {
            wp_die(__('Ogiltig förfrågan (nonce).', 'macken-wxr-export'));
        }

        $term_id = isset($_POST['category_id']) ? intval($_POST['category_id']) : 0;
        $include_trash = isset($_POST['include_trash']) && $_POST['include_trash'] === '1';

        if ($term_id <= 0) {
            wp_die('Välj en kategori.');
        }

        $this->do_export($term_id, $include_trash);
        exit;
    }

    private function do_export(int $term_id, bool $include_trash) {
        @set_time_limit(0);
        ignore_user_abort(true);

        $statuses = ['publish','draft','pending','future','private','inherit'];
        if ($include_trash) $statuses[] = 'trash';

        $term = get_term($term_id, 'category');
        if (!$term || is_wp_error($term)) {
            wp_die('Ogiltig kategori.');
        }

        // Collect posts
        $paged     = 1;
        $per_page  = 500;
        $all_posts = [];

        do {
            $q = new WP_Query([
                'post_type'      => 'post',
                'post_status'    => $statuses,
                'posts_per_page' => $per_page,
                'paged'          => $paged,
                'tax_query'      => [[
                    'taxonomy' => 'category',
                    'field'    => 'term_id',
                    'terms'    => $term_id,
                ]],
                'orderby'        => 'date',
                'order'          => 'ASC',
                'fields'         => 'ids',
                'no_found_rows'  => false,
            ]);

            if ($q->have_posts()) $all_posts = array_merge($all_posts, $q->posts);
            $max_pages = $q->max_num_pages ?: 0;
            $paged++;
            wp_reset_postdata();
        } while ($max_pages && $paged <= $max_pages);

        // Also collect attachments for these posts
        $attachments = [];
        if (!empty($all_posts)) {
            $att_q = new WP_Query([
                'post_type'      => 'attachment',
                'post_status'    => ['inherit','private','trash'],
                'posts_per_page' => -1,
                'post_parent__in'=> $all_posts,
                'fields'         => 'ids',
                'no_found_rows'  => true,
            ]);
            if ($att_q->have_posts()) $attachments = $att_q->posts;
            wp_reset_postdata();
        }

        $filename = sprintf(
            'macken-export-%s-%s.xml',
            sanitize_title($term->slug),
            date_i18n('Ymd-His')
        );

        $xml = $this->generate_wxr($term, $all_posts, $attachments);
        $length = strlen($xml);

        // Clean any output buffers
        while (ob_get_level()) { @ob_end_clean(); }

        status_header(200);
        nocache_headers();
        header('Content-Description: File Transfer');
        header('Content-Type: application/xml; charset=' . get_option('blog_charset'));
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Transfer-Encoding: binary');
        header('Content-Length: ' . $length);

        echo $xml;
        exit;
    }

    private function cdata($str) {
        $str = (string)$str;
        $str = str_replace(']]>', ']]]]><![CDATA[>', $str);
        return "<![CDATA[$str]]>";
    }

    private function xml($str) {
        return esc_html($str);
    }

    private function generate_wxr(WP_Term $term, array $post_ids, array $attachment_ids) {
        $site_name = get_bloginfo('name');
        $site_url  = home_url('/');
        $wxr_ver   = '1.2';

        // Authors present
        $authors = [];
        foreach ($post_ids as $pid) {
            $post = get_post($pid);
            if (!$post) continue;
            $uid = (int)$post->post_author;
            if (!isset($authors[$uid])) {
                $u = get_userdata($uid);
                if ($u) {
                    $authors[$uid] = [
                        'id'       => $uid,
                        'login'    => $u->user_login,
                        'email'    => $u->user_email,
                        'display'  => $u->display_name,
                        'first'    => $u->first_name,
                        'last'     => $u->last_name,
                    ];
                }
            }
        }

        // Terms used
        $used_terms = [
            'category' => [],
            'post_tag' => [],
        ];
        foreach ($post_ids as $pid) {
            foreach (['category','post_tag'] as $tax) {
                $ts = wp_get_post_terms($pid, $tax);
                if (!is_wp_error($ts)) {
                    foreach ($ts as $t) {
                        $used_terms[$tax][$t->term_id] = $t;
                    }
                }
            }
        }

        ob_start();
        echo '<?xml version="1.0" encoding="'.get_option('blog_charset').'"?>'."\n";
        ?>
<rss version="2.0"
    xmlns:excerpt="http://wordpress.org/export/1.2/excerpt/"
    xmlns:content="http://purl.org/rss/1.0/modules/content/"
    xmlns:wfw="http://wellformedweb.org/CommentAPI/"
    xmlns:dc="http://purl.org/dc/elements/1.1/"
    xmlns:wp="http://wordpress.org/export/1.2/"
>
<channel>
    <title><?php echo $this->xml($site_name); ?></title>
    <link><?php echo $this->xml($site_url); ?></link>
    <description><?php echo $this->xml(get_bloginfo('description')); ?></description>
    <pubDate><?php echo $this->xml(date(DATE_RSS)); ?></pubDate>
    <language><?php echo $this->xml(get_bloginfo('language')); ?></language>
    <wp:wxr_version><?php echo $this->xml($wxr_ver); ?></wp:wxr_version>
    <wp:base_site_url><?php echo $this->xml(network_home_url('/')); ?></wp:base_site_url>
    <wp:base_blog_url><?php echo $this->xml(home_url('/')); ?></wp:base_blog_url>

    <!-- Kategori som exporten utgår från -->
    <wp:category>
        <wp:term_id><?php echo (int)$term->term_id; ?></wp:term_id>
        <wp:category_nicename><?php echo $this->xml($term->slug); ?></wp:category_nicename>
        <wp:cat_name><?php echo $this->cdata($term->name); ?></wp:cat_name>
        <?php if (!empty($term->description)): ?>
        <wp:category_description><?php echo $this->cdata($term->description); ?></wp:category_description>
        <?php endif; ?>
    </wp:category>

    <?php foreach ($used_terms['category'] as $t): ?>
    <wp:category>
        <wp:term_id><?php echo (int)$t->term_id; ?></wp:term_id>
        <wp:category_nicename><?php echo $this->xml($t->slug); ?></wp:category_nicename>
        <wp:cat_name><?php echo $this->cdata($t->name); ?></wp:cat_name>
        <?php if ($t->parent): ?>
        <wp:category_parent><?php echo $this->xml(get_term_field('slug', $t->parent, 'category')); ?></wp:category_parent>
        <?php endif; ?>
        <?php if (!empty($t->description)): ?>
        <wp:category_description><?php echo $this->cdata($t->description); ?></wp:category_description>
        <?php endif; ?>
    </wp:category>
    <?php endforeach; ?>

    <?php foreach ($used_terms['post_tag'] as $t): ?>
    <wp:tag>
        <wp:term_id><?php echo (int)$t->term_id; ?></wp:term_id>
        <wp:tag_slug><?php echo $this->xml($t->slug); ?></wp:tag_slug>
        <wp:tag_name><?php echo $this->cdata($t->name); ?></wp:tag_name>
        <?php if (!empty($t->description)): ?>
        <wp:tag_description><?php echo $this->cdata($t->description); ?></wp:tag_description>
        <?php endif; ?>
    </wp:tag>
    <?php endforeach; ?>

    <?php foreach ($authors as $a): ?>
    <wp:author>
        <wp:author_id><?php echo (int)$a['id']; ?></wp:author_id>
        <wp:author_login><?php echo $this->cdata($a['login']); ?></wp:author_login>
        <wp:author_email><?php echo $this->cdata($a['email']); ?></wp:author_email>
        <wp:author_display_name><?php echo $this->cdata($a['display']); ?></wp:author_display_name>
        <wp:author_first_name><?php echo $this->cdata($a['first']); ?></wp:author_first_name>
        <wp:author_last_name><?php echo $this->cdata($a['last']); ?></wp:author_last_name>
    </wp:author>
    <?php endforeach; ?>

    <?php
    // Posts
    foreach ($post_ids as $pid):
        $p = get_post($pid);
        if (!$p) continue;

        $post_link   = get_permalink($p);
        $post_guid   = $p->guid;
        $creator     = get_the_author_meta('login', $p->post_author);
        $post_date   = mysql2date('Y-m-d H:i:s', $p->post_date, false);
        $post_date_gmt = mysql2date('Y-m-d H:i:s', $p->post_date_gmt, false);

        $cats = wp_get_post_terms($p->ID, 'category');
        $tags = wp_get_post_terms($p->ID, 'post_tag');

        $meta = get_post_meta($p->ID);
        ?>
    <item>
        <title><?php echo $this->cdata(get_the_title($p)); ?></title>
        <link><?php echo $this->xml($post_link); ?></link>
        <pubDate><?php echo $this->xml(mysql2date(DATE_RSS, $p->post_date)); ?></pubDate>
        <dc:creator><?php echo $this->cdata($creator); ?></dc:creator>
        <guid isPermaLink="false"><?php echo $this->xml($post_guid); ?></guid>
        <description></description>
        <content:encoded><?php echo $this->cdata($p->post_content); ?></content:encoded>
        <excerpt:encoded><?php echo $this->cdata($p->post_excerpt); ?></excerpt:encoded>
        <wp:post_id><?php echo (int)$p->ID; ?></wp:post_id>
        <wp:post_date><?php echo $this->xml($post_date); ?></wp:post_date>
        <wp:post_date_gmt><?php echo $this->xml($post_date_gmt); ?></wp:post_date_gmt>
        <wp:comment_status><?php echo $this->xml($p->comment_status); ?></wp:comment_status>
        <wp:ping_status><?php echo $this->xml($p->ping_status); ?></wp:ping_status>
        <wp:post_name><?php echo $this->xml($p->post_name); ?></wp:post_name>
        <wp:status><?php echo $this->xml($p->post_status); ?></wp:status>
        <wp:post_parent><?php echo (int)$p->post_parent; ?></wp:post_parent>
        <wp:menu_order><?php echo (int)$p->menu_order; ?></wp:menu_order>
        <wp:post_type>post</wp:post_type>
        <wp:post_password><?php echo $this->xml($p->post_password); ?></wp:post_password>
        <wp:is_sticky><?php echo is_sticky($p->ID) ? 1 : 0; ?></wp:is_sticky>
        <?php if (!empty($p->post_modified)) : ?>
        <wp:post_modified><?php echo $this->xml(mysql2date('Y-m-d H:i:s', $p->post_modified, false)); ?></wp:post_modified>
        <wp:post_modified_gmt><?php echo $this->xml(mysql2date('Y-m-d H:i:s', $p->post_modified_gmt, false)); ?></wp:post_modified_gmt>
        <?php endif; ?>

        <?php if (!is_wp_error($cats)) { foreach ($cats as $c) { ?>
        <category domain="category" nicename="<?php echo $this->xml($c->slug); ?>"><?php echo $this->cdata($c->name); ?></category>
        <?php }} ?>
        <?php if (!is_wp_error($tags)) { foreach ($tags as $t) { ?>
        <category domain="post_tag" nicename="<?php echo $this->xml($t->slug); ?>"><?php echo $this->cdata($t->name); ?></category>
        <?php }} ?>

        <?php
        if (is_array($meta)) {
            foreach ($meta as $m_key => $m_vals) {
                foreach ((array)$m_vals as $m_val) {
                    $is_serialized = is_serialized($m_val);
                    $value = $is_serialized ? base64_encode($m_val) : $m_val;
        ?>
        <wp:postmeta>
            <wp:meta_key><?php echo $this->cdata($m_key); ?></wp:meta_key>
            <wp:meta_value><?php echo $this->cdata($value); ?></wp:meta_value>
        </wp:postmeta>
        <?php
                }
            }
        }
        ?>
    </item>
    <?php endforeach; ?>

    <?php
    // Attachments (children of exported posts)
    foreach ($attachment_ids as $aid):
        $a = get_post($aid);
        if (!$a) continue;
        $url  = wp_get_attachment_url($aid);
        $meta = wp_get_attachment_metadata($aid);
        $title = get_the_title($aid);
        $creator = get_the_author_meta('login', $a->post_author);
        $post_date   = mysql2date('Y-m-d H:i:s', $a->post_date, false);
        $post_date_gmt = mysql2date('Y-m-d H:i:s', $a->post_date_gmt, false);
        ?>
    <item>
        <title><?php echo $this->cdata($title); ?></title>
        <link><?php echo $this->xml($url); ?></link>
        <pubDate><?php echo $this->xml(mysql2date(DATE_RSS, $a->post_date)); ?></pubDate>
        <dc:creator><?php echo $this->cdata($creator); ?></dc:creator>
        <guid isPermaLink="false"><?php echo $this->xml($a->guid); ?></guid>
        <description></description>
        <content:encoded><?php echo $this->cdata(''); ?></content:encoded>
        <excerpt:encoded><?php echo $this->cdata(''); ?></excerpt:encoded>
        <wp:post_id><?php echo (int)$a->ID; ?></wp:post_id>
        <wp:post_date><?php echo $this->xml($post_date); ?></wp:post_date>
        <wp:post_date_gmt><?php echo $this->xml($post_date_gmt); ?></wp:post_date_gmt>
        <wp:comment_status>closed</wp:comment_status>
        <wp:ping_status>closed</wp:ping_status>
        <wp:post_name><?php echo $this->xml($a->post_name); ?></wp:post_name>
        <wp:status><?php echo $this->xml($a->post_status); ?></wp:status>
        <wp:post_parent><?php echo (int)$a->post_parent; ?></wp:post_parent>
        <wp:menu_order>0</wp:menu_order>
        <wp:post_type>attachment</wp:post_type>
        <wp:post_password></wp:post_password>
        <wp:is_sticky>0</wp:is_sticky>
        <wp:attachment_url><?php echo $this->xml($url); ?></wp:attachment_url>
        <?php if (!empty($meta)) { ?>
        <wp:postmeta>
            <wp:meta_key>_wp_attachment_metadata</wp:meta_key>
            <wp:meta_value><?php echo $this->cdata(maybe_serialize($meta)); ?></wp:meta_value>
        </wp:postmeta>
        <?php } ?>
        <?php
        $alt = get_post_meta($a->ID, '_wp_attachment_image_alt', true);
        if ($alt !== '') { ?>
        <wp:postmeta>
            <wp:meta_key>_wp_attachment_image_alt</wp:meta_key>
            <wp:meta_value><?php echo $this->cdata($alt); ?></wp:meta_value>
        </wp:postmeta>
        <?php } ?>
    </item>
    <?php endforeach; ?>

</channel>
</rss>
<?php
        return ob_get_clean();
    }
}

new Macken_WXR_Export();
